// ModuleResolver - logic for resolving modules in node, w/o deps
// supposed to be inlined, not loaded via require/etc

var ModuleResolver  = {
    
    requires    : [ 'fileExists', 'getFileAsJSON', 'getCoreModule', 'addAdditionalDependency' ],
    
    has : {
        coreModules         : { required : true }
    },
    
    
    methods : {
        
        cleanupPath : function (path) {
            path            = path.replace(/\/+/g, '/')
            
            // XXX todo proper parent dir filter (right now breaks for folder names with dots)
            var currentDir  = /(^|\/)\.\//
            var parentDir   = /[^/.]+\/\.\.\//
            
            while (currentDir.test(path))   path = path.replace(currentDir, '$1')
            while (parentDir.test(path))    path = path.replace(parentDir, '')
            
            return path
        },
        
        
        resolveAsDir : function (requireStr, dependencies) {
            // strip trailing slash
            requireStr      = requireStr.replace(/\/+$/, '')
            requireStr      = this.cleanupPath(requireStr)
            
            var packageFileName = requireStr + '/package.json'
            
            if (this.fileExists(packageFileName)) {
                var packageJSON = this.getFileAsJSON(packageFileName)
                
                var main        = packageJSON.main
                
                // if `main` key presents - rootResolve it as file
                if (main)       {
                    var res = this.resolveAsFile(requireStr + '/' + main)
                    
                    if (res) this.addAdditionalDependency(dependencies, packageFileName)
                    
                    return res
                }
            }
            
            // and finally try to load "index"
            return this.resolveAsFile(requireStr + '/index')
        },
        
        
        resolveAsFile : function (requireStr) {
            requireStr      = this.cleanupPath(requireStr)
            
            if (this.fileExists(requireStr)) return requireStr 
            
            if (this.fileExists(requireStr + '.js')) return requireStr + '.js'
            
            return null
        },
        
        
        resolveAsModule : function (requireStr, baseDir, dependencies) {
            baseDir         = baseDir.replace(/\/+$/, '')
            
            var parts       = baseDir.split(/\//)
            var dirs        = []
            
            while (parts.length) {
                var segment     = parts.pop()
                
                // XXX the /node_modules in root will be ignored
                if (segment != 'node_modules') {
                    var dir     = [].concat(parts, segment, 'node_modules')
                    
                    dirs.push(dir.join('/'))
                }
            }
            
            dirs.push('node_modules')
            
            for (var i = 0; i < dirs.length; i++) {
                
                var resolved    = this.resolveAsDir(dirs[ i ] + '/' + requireStr, dependencies)
                
                if (resolved) return resolved
            }
            
            return null
        },
        
        
        // as part of the resolving procedure, one may need to report about additional "hidden" dependencies
        // that are not directly referenced by any file, but are used in the node modules resolution logic
        // notable those are "package.json" files 
        // when such additional dependency will be discovered the role will "register" it with `addAdditionalDependency`
        // call, passing the `dependencies` argument and file name to it
        resolve : function (requireStr, baseDir, dependencies) {
            if (this.coreModules[ requireStr ]) return this.getCoreModule(requireStr)
            
            baseDir     = (baseDir || '.').replace(/\/+$/, '')
            
            if (/^\//.test(requireStr))             return this.resolveAsFile(requireStr) || this.resolveAsDir(requireStr, dependencies)
            if (/^\.\.?\//.test(requireStr))  {
                var path    = baseDir + '/' + requireStr
                
                return this.resolveAsFile(path) || this.resolveAsDir(path, dependencies)
            }

            return this.resolveAsModule(requireStr, baseDir, dependencies)
        }
    }
}
